<?php

/**
 * Class TeamModel. Equipos
 */
class TeamModel extends CI_Model


{
    const TABLE_NAME = 'Equipo';
    const PROVINCE = 'provincia';
    const AUTONOMOUS_COMMUNITY = 'autonomia';
    const PAISISOA3 = 'paisISOA3';

    /**
     * TeamModel constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Funcion que devuelve todos los equipos dentro de un rango de datos
     * para la paginacion
     * @param $perpage numero de equipos por pagina
     * @param $segment_uri Desplazamiento
     * @return Equipo [] con todos sus datos
     */
    public function getTeamsBy($perpage, $segment_uri)
    {
        $this->db->order_by('nombre', 'asc');
        $query = $this->db->get(self::TABLE_NAME, $perpage, $segment_uri);

        return $query->result();
    }

    /**
     * Funcion que devuelve todos los equipos
     * @return Equipo []
     */
    public function getAllTeams()
    {
        $sql = 'SELECT E.* FROM Equipo E ORDER BY E.nombre ASC';
        $query = $this->db->query($sql);
        return $query->result();
    }

    /**
     * Funcion que indica si existe un equipo con el nombre pasado por parametro
     * @param $name Nombre del equipo del que se quiere averiguar si existe o no
     * @return bool true en caso de que exista un equipo con el nombre pasado por parametro
     * false en caso contrario
     */
    public function existTeam($name)
    {
        $sql = 'SELECT E.* FROM Equipo E WHERE E.nombre = ?';
        $query = $this->db->query($sql, array($name));
        return $query->num_rows() != 0;
    }

    /**
     * Funcion utilizada para crear un nuevo equipo
     * @param $nombre Nombre del equipo
     * @param $directorTecnico Nombre del director tecnico
     * @param $pais Pais donde tiene sede el equipo
     * @param $comunidadAutonoma Comunidad autonoma
     * @param $provincia Provincia
     * @return true si se crea correctamente, false en caso contrario
     */
    public function createTeam($nombre, $directorTecnico, $pais, $comunidadAutonoma, $provincia)
    {
        $team = array(
            'nombre' => $nombre,
            'directorTecnico' => $directorTecnico,
            'paisISOA3' => $pais,
            'autonomia' => $comunidadAutonoma,
            'provincia' => $provincia

        );

        return $this->db->insert(self::TABLE_NAME, $team);
    }

    /**
     * Funcion para eliminar un equipo
     * @param $teamName Identificador del equipo que se quiere eliminar
     * @return true si se elimina correctamente, false en caso contrario
     */
    public function deleteTeam($teamName)
    {
        $this->db->where('nombre', $teamName);
        return $this->db->delete(self::TABLE_NAME);
    }

    /**
     * Funcion para ordenar los equipos en funcion de un campo
     * @param $condition
     * @return Equipo [] ordenados ascendentemente por la condicion
     */
    public function sortTeamsBy($condition)
    {
        $this->db->from(self::TABLE_NAME);
        $this->db->order_by($condition, "asc");
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Funcion para actualizar los datos de un equipo
     * @param $nombre Nombre del equipo
     * @param $directorTecnico Nombre del director tecnico
     * @param $pais Pais
     * @param $comunidadAutonoma Comunidad autonoma
     * @param $provincia Provincia
     * @return true si se actualiza correctamente, false en caso contrario
     */
    public function updateTeam($nombre, $directorTecnico, $pais, $comunidadAutonoma, $provincia)
    {
        $data = array(
            'nombre' => $nombre,
            'directorTecnico' => $directorTecnico,
            'paisISOA3' => $pais,
            'autonomia' => $comunidadAutonoma,
            'provincia' => $provincia

        );

        $this->db->where('nombre', $nombre);
        return $this->db->update(self::TABLE_NAME, $data);
    }
}